'use strict';

/* --------------------------------------------------------------
 datatable_responsive_columns.js 2017-03-08
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2017 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * ## Enable DataTable Responsive Columns
 *
 * This module will enable the responsive columns functionality which will resize the columns until a minimum
 * width is reach. Afterwards the columns will be hidden and the content will be displayed by through an icon
 * tooltip.
 *
 * ### Options
 *
 * **Initial Visibility Toggle Selector | `data-data_relative_columns-visibility-toggle-selector` | String | Optional**
 *
 * Provide a selector relative to each thead > tr element in order to hide the column on page load and then show it
 * again once the responsive widths have been calculated. The provided selector must point to the biggest column in
 * order to avoid broken displays till the table becomes responsive.
 *
 * @module Admin/Extensions/data_relative_columns
 */
gx.extensions.module('datatable_responsive_columns', [jse.source + '/vendor/qtip2/jquery.qtip.min.css', jse.source + '/vendor/qtip2/jquery.qtip.min.js'], function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLES
    // ------------------------------------------------------------------------

    /**
     * Module Selector
     *
     * @type {jQuery}
     */

    var $this = $(this);

    /**
     * Default Options
     *
     * @type {Object}
     */
    var defaults = {
        visibilityToggleSelector: '[data-column-name="actions"]'
    };

    /**
     * Final Options
     *
     * @type {Object}
     */
    var options = $.extend(true, {}, defaults, data);

    /**
     * Module Instance
     *
     * @type {Object}
     */
    var module = {};

    /**
     * DataTable Initialization Columns
     *
     * @type {Array}
     */
    var columnDefinitions = void 0;

    /**
     * Width Factor Sum
     *
     * @type {Number}
     */
    var widthFactorSum = void 0;

    // ------------------------------------------------------------------------
    // FUNCTIONS
    // ------------------------------------------------------------------------

    /**
     * Update empty table "colspan" attribute.
     *
     * This method will keep the empty table row width in sync with the table width.
     */
    function _updateEmptyTableColSpan() {
        if ($this.find('.dataTables_empty').length > 0) {
            var colspan = $this.find('thead:first tr:first .actions').index() + 1 - $this.find('thead:first tr:first th.hidden').length;
            $this.find('.dataTables_empty').attr('colspan', colspan);
        }
    }

    /**
     * Add hidden columns content icon to actions cell of a single row.
     *
     * Call this method only if you are sure there is no icon previously set (runs faster).
     *
     * @param {jQuery} $tr
     */
    function _addHiddenColumnsContentIcon($tr) {
        $tr.find('td.actions div:first').prepend('<i class="fa fa-ellipsis-h meta-icon hidden-columns-content"></i>');
    }

    /**
     * Get the cell content.
     *
     * This method will also search for child input and select elements and return the appropriate content.
     *
     * @param {jQuery} $td Table cell to be examined.
     *
     * @return {String}
     */
    function _getCellContent($td) {
        if ($td.find('select').length) {
            return $td.find('select option:selected').text();
        } else if ($td.find('input:text').length) {
            return $td.find('input:text').val();
        } else if ($td.find('input:checkbox').length) {
            return $td.find('input:checkbox').prop('checked') ? '✔' : '✘';
        } else {
            return $td.text();
        }
    }

    /**
     * Generates and sets the tooltip content for the hidden columns content.
     *
     * @param {jQuery} $tr The current row selector.
     */
    function _generateHiddenColumnsContent($tr) {
        var hiddenColumnContentHtml = '';

        $tr.find('td.hidden').each(function (index, td) {
            hiddenColumnContentHtml += $this.find('thead:first tr:first th:eq(' + $(td).index() + ')').text() + ': ' + _getCellContent($(td)) + '<br/>';
        });

        $tr.find('.hidden-columns-content').qtip({
            content: hiddenColumnContentHtml,
            style: {
                classes: 'gx-qtip info'
            },
            hide: {
                fixed: true,
                delay: 300
            }
        });
    }

    /**
     * Hide DataTable Columns
     *
     * This method is part of the responsive tables solution.
     *
     * @param {jQuery} $targetWrapper Target datatable instance wrapper div.
     * @param {jQuery} $firstHiddenColumn The first hidden column (first column with the .hidden class).
     */
    function _hideColumns($targetWrapper, $firstHiddenColumn) {
        var $lastVisibleColumn = $firstHiddenColumn.length !== 0 ? $firstHiddenColumn.prev() : $this.find('thead:first th.actions').prev();

        if ($lastVisibleColumn.hasClass('hidden') || $lastVisibleColumn.index() === 0) {
            return; // First column or already hidden, do not continue.
        }

        // Show hidden column content icon.
        if ($this.find('.hidden-columns-content').length === 0) {
            $this.find('tbody tr').each(function (index, tr) {
                _addHiddenColumnsContentIcon($(tr));
            });
        }

        // Hide the last visible column.
        $this.find('tr').each(function (index, tr) {
            $(tr).find('th:eq(' + $lastVisibleColumn.index() + '), td:eq(' + $lastVisibleColumn.index() + ')').addClass('hidden');

            // Generate the hidden columns content.
            _generateHiddenColumnsContent($(tr));
        });

        _updateEmptyTableColSpan();

        // If there are still columns which don't fit within the viewport, hide them.
        if ($targetWrapper.width() < $this.width() && $lastVisibleColumn.index() > 1) {
            _toggleColumnsVisibility();
        }
    }

    /**
     * Show DataTable Columns
     *
     * This method is part of the responsive tables solution.
     *
     * @param {jQuery} $targetWrapper Target datatable instance wrapper div.
     * @param {jQuery} $firstHiddenColumn The first hidden column (first column with the .hidden class).
     */
    function _showColumns($targetWrapper, $firstHiddenColumn) {
        if ($firstHiddenColumn.length === 0) {
            return;
        }

        var firstHiddenColumnWidth = parseInt($firstHiddenColumn.css('min-width'));
        var tableMinWidth = 0;

        // Calculate the table min width by each column min width.
        $this.find('thead:first tr:first th').each(function (index, th) {
            if (!$(th).hasClass('hidden')) {
                tableMinWidth += parseInt($(th).css('min-width'));
            }
        });

        // Show the first hidden column.
        if (tableMinWidth + firstHiddenColumnWidth <= $targetWrapper.outerWidth()) {
            $this.find('tr').each(function (index, tr) {
                $(tr).find('th:eq(' + $firstHiddenColumn.index() + '), td:eq(' + $firstHiddenColumn.index() + ')').removeClass('hidden');

                _generateHiddenColumnsContent($(tr));
            });

            _updateEmptyTableColSpan();

            // Hide hidden column content icon.
            if ($this.find('thead:first tr:first th.hidden').length === 0) {
                $this.find('.hidden-columns-content').remove();
            }

            // If there are still columns which would fit fit within the viewport, show them.
            var newTableMinWidth = tableMinWidth + firstHiddenColumnWidth + parseInt($firstHiddenColumn.next('.hidden').css('min-width'));

            if (newTableMinWidth <= $targetWrapper.outerWidth() && $firstHiddenColumn.next('.hidden').length !== 0) {
                _toggleColumnsVisibility();
            }
        }
    }

    /**
     * Toggle column visibility depending the window size.
     */
    function _toggleColumnsVisibility() {
        var $targetWrapper = $this.parent();
        var $firstHiddenColumn = $this.find('thead:first th.hidden:first');

        if ($targetWrapper.width() < $this.width()) {
            _hideColumns($targetWrapper, $firstHiddenColumn);
        } else {
            _showColumns($targetWrapper, $firstHiddenColumn);
        }
    }

    /**
     * Calculate and set the relative column widths.
     *
     * The relative width calculation works with a width-factor system where each column preserves a
     * specific amount of the table width.
     *
     * This factor is not defining a percentage, rather only a width-volume. Percentage widths will not
     * work correctly when the table has fewer columns than the original settings.
     */
    function _applyRelativeColumnWidths() {
        $this.find('thead:first tr:first th').each(function () {
            var _this = this;

            if ($(this).css('display') === 'none') {
                return true;
            }

            var currentColumnDefinition = void 0;

            columnDefinitions.forEach(function (columnDefinition) {
                if (columnDefinition.name === $(_this).data('columnName')) {
                    currentColumnDefinition = columnDefinition;
                }
            });

            if (currentColumnDefinition && currentColumnDefinition.widthFactor) {
                var index = $(this).index();
                var width = Math.round(currentColumnDefinition.widthFactor / widthFactorSum * 100 * 100) / 100;
                $this.find('thead').each(function (i, thead) {
                    $(thead).find('tr').each(function (i, tr) {
                        $(tr).find('th').eq(index).css('width', width + '%');
                    });
                });
            }
        });
    }

    /**
     * Applies the column width if the current column width is smaller.
     */
    function _applyMinimumColumnWidths() {
        $this.find('thead:first tr:first th').each(function (index) {
            var _this2 = this;

            if ($(this).css('display') === 'none') {
                return true;
            }

            var currentColumnDefinition = void 0;

            columnDefinitions.forEach(function (columnDefinition) {
                if (columnDefinition.name === $(_this2).data('columnName')) {
                    currentColumnDefinition = columnDefinition;
                }
            });

            if (!currentColumnDefinition) {
                return true;
            }

            var currentWidth = $(this).outerWidth();
            var definitionMinWidth = parseInt(currentColumnDefinition.minWidth);

            if (currentWidth < definitionMinWidth) {
                // Force the correct column min-widths for all thead columns.
                $this.find('thead').each(function (i, thead) {
                    $(thead).find('tr').each(function (i, tr) {
                        $(tr).find('th').eq(index).outerWidth(definitionMinWidth).css('max-width', definitionMinWidth).css('min-width', definitionMinWidth);
                    });
                });
            }
        });
    }

    /**
     * On DataTable Draw Event
     */
    function _onDataTableDraw() {
        // Wait until the contents of the table are rendered. DataTables will sometimes fire the draw event
        // even before the td elements are rendered in the browser.
        var interval = setInterval(function () {
            if ($this.find('tbody tr:last td.actions').length === 1) {
                _applyRelativeColumnWidths();
                _applyMinimumColumnWidths();
                _toggleColumnsVisibility();

                // Hide the tbody cells depending on whether the respective <th> element is hidden.
                $this.find('thead:first tr:first th').each(function (index, th) {
                    if ($(th).hasClass('hidden')) {
                        $this.find('tbody tr').each(function (i, tr) {
                            $(tr).find('td:eq(' + index + ')').addClass('hidden');
                        });
                    }
                });

                // Add the hidden columns icon if needed.
                if ($this.find('thead th.hidden').length) {
                    $this.find('tbody tr').each(function (index, tr) {
                        if ($(tr).find('.hidden-columns-content').length) {
                            return true;
                        }
                        _addHiddenColumnsContentIcon($(tr));
                        _generateHiddenColumnsContent($(tr));
                    });
                }

                clearInterval(interval);
            }
        }, 500);
    }

    /**
     * On Window Resize Event
     */
    function _onWindowResize() {
        $this.find('thead.fixed').outerWidth($this.outerWidth());
        _applyRelativeColumnWidths();
        _applyMinimumColumnWidths();
        _toggleColumnsVisibility();
    }

    /**
     * On DataTable Initialize Event
     */
    function _onDataTableInit() {
        $this.find(options.visibilityToggleSelector).show();
        _updateEmptyTableColSpan();

        columnDefinitions = $this.DataTable().init().columns;
        widthFactorSum = 0;

        columnDefinitions.forEach(function (columnDefinition) {
            widthFactorSum += columnDefinition.widthFactor || 0;
        });

        $this.on('draw.dt', _onDataTableDraw);
        $(window).on('resize', _onWindowResize);

        _onWindowResize();
    }

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    module.init = function (done) {
        $this.on('init.dt', _onDataTableInit);

        $(window).on('JSENGINE_INIT_FINISHED', function () {
            if ($this.DataTable().ajax.json() !== undefined) {
                _onDataTableInit();
            }
        });

        $this.find(options.visibilityToggleSelector).hide();

        done();
    };

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
